import asyncio

from devicepilot.pylog.pylogger import PyLogger

from devicepilot.hw_abstraction.hal import HALBase
from devicepilot.hw_abstraction.node_ctrl import Node
from devicepilot.hw_abstraction.mover_ctrl import Mover
from devicepilot.hw_abstraction.dual_mover_ctrl import DualMover
from devicepilot.hw_abstraction.serial_ctrl import Serial
from devicepilot.hw_abstraction.barcode_reader_ctrl import BarcodeReader

from config_enum import hal_enum
from hw_abstraction.eef_measurement_unit_ctrl import EEFMeasurementUnit
from urpc.node_io import FMBDigitalOutput
from urpc.node_io import EEFAnalogOutput


class HAL(HALBase):

    def __init__(self, test_stage=''):
        HALBase.__init__(self)

        # Test Stage Hardware
        self.test_stage = test_stage if test_stage != '' else self.get_config(hal_enum.HAL.TestStage)
        match self.test_stage.lower():
            case 'pmt_fi_adjust':
                PyLogger.logger.info("Setup PMT-FI Adjustment Stage Hardware...")
                self.fmb: Node = self.hw_controller_factory(Node, hal_enum.PMTFIAdjust.FMB)
                self.mc6: Node = self.hw_controller_factory(Node, hal_enum.PMTFIAdjust.MC6)
                self.eef: Node = self.hw_controller_factory(Node, hal_enum.PMTFIAdjust.EEF)
                self.measurement_unit: EEFMeasurementUnit = self.hw_controller_factory(EEFMeasurementUnit, hal_enum.PMTFIAdjust.MeasurementUnit)
                self.od_filter_wheel1: Mover = self.hw_controller_factory(Mover, hal_enum.PMTFIAdjust.ODFilterWheel1)
                self.od_filter_wheel2: Mover = self.hw_controller_factory(Mover, hal_enum.PMTFIAdjust.ODFilterWheel2)
                self.detector_aperture_slider1: Mover = self.hw_controller_factory(Mover, hal_enum.PMTFIAdjust.DetectorApertureSlider1)
            case 'pmt_adjust':
                PyLogger.logger.info("Setup PMT Adjustment Stage Hardware...")
                self.fmb: Node = self.hw_controller_factory(Node, hal_enum.PMTAdjust.FMB)
                self.mc6: Node = self.hw_controller_factory(Node, hal_enum.PMTFIAdjust.MC6)
                self.eef: Node = self.hw_controller_factory(Node, hal_enum.PMTAdjust.EEF)
                self.measurement_unit: EEFMeasurementUnit = self.hw_controller_factory(EEFMeasurementUnit, hal_enum.PMTAdjust.MeasurementUnit)
            case 'base_tester':
                PyLogger.logger.info("Setup Base Tester Hardware...")
                self.fmb: Node = self.hw_controller_factory(Node, hal_enum.BaseTester.FMB)
                self.mc6: Node = self.hw_controller_factory(Node, hal_enum.BaseTester.MC6)
                self.eef: Node = self.hw_controller_factory(Node, hal_enum.BaseTester.EEF)
                self.mc6m1: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.MC6M1)
                self.mc6m2: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.MC6M2)
                self.mc6m3: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.MC6M3)
                self.mc6m4: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.MC6M4)
                self.mc6m5: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.MC6M5)
                self.mc6m6: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.MC6M6)
                self.eefst: DualMover = self.hw_controller_factory(DualMover, hal_enum.BaseTester.EEFST)
                self.eefm3: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.EEFM3)
                self.eefm4: Mover = self.hw_controller_factory(Mover, hal_enum.BaseTester.EEFM4)
                self.trf: Serial = self.hw_controller_factory(Serial, hal_enum.BaseTester.TRF)
                self.bcr1: BarcodeReader = self.hw_controller_factory(BarcodeReader, hal_enum.BaseTester.BCR1)
                self.bcr2: BarcodeReader = self.hw_controller_factory(BarcodeReader, hal_enum.BaseTester.BCR2)
            case _:
                PyLogger.logger.error(f"Test stage '{self.test_stage}' not supported.")

    async def startup_hardware(self):
        """
        First start FMB firmware and power up the Base Tester, then start the firmware of the other nodes.

        The Base Tester has a different logic for the PSU ON signal compared to the Nexus.
        The PSU ON signal is active low,. It can only switch the 24V supply of the MC6 and EEF, but not the FMB itself.
        Also the DC OK signal is not used in the Base Tester.
        """
        match self.test_stage.lower():
            case 'pmt_fi_adjust' | 'pmt_adjust' | 'base_tester':
                await self.fmb.start_firmware()
                await self.fmb.set_digital_output(FMBDigitalOutput.PSUON, 0)    # Power = ON (active low)
                await asyncio.sleep(0.1)
                await self.gather_but_dont_cancel(
                    self.mc6.start_firmware(),
                    self.eef.start_firmware(),
                )
                await self.eef.set_analog_output(EEFAnalogOutput.FLASHFAN, 1.0) # Turn on the flashlamp fan
            case _:
                super().startup_hardware()

    async def shutdown_hardware(self):
        """
        Reset the MC6 and EEF, then power off the Base Tester by resetting the FMB.
        A side effect of the Bese Testers power supply logic is that a reset of the FMB will also power off the MC6 and EEF.
        """
        try:
            await self.mc6.reset()
        except Exception as ex:
            PyLogger.logger.warning(f"Failed to reset MC6: {ex}")

        try:
            await self.eef.reset()
        except Exception as ex:
            PyLogger.logger.warning(f"Failed to reset EEF: {ex}")

        await asyncio.sleep(0.1)
        await self.fmb.reset()
